function ind = findtrial(Nt, maxNt, Ns)

% FINDTRIAL Index responses corresponding to trials.
%
% USAGE
%   IND = FINDTRIAL(NT, MAXNT, NS)
%
% INPUT
%   NT    - number of trials per stimulus array.
%   MAXNT - max number of trials (must be equal to max(NT) and size(NT,2))
%   NS    - number of stimuli (must be equal to length(NT))
%
% OUTPUT
%   TRIALINDX returns the position of the trials in the response matrix R
%   (see ENTROPY and INFORMATION) ignoring the first dimension
%   (dimensionality of the response).
%
% EXAMPLE
%   Let's consider the response matrix in the figure below (letters are
%   used to represent undefined values):
%
%                             ----------------- 
%                             | o | q | s | - |
%                             |---------------|
%                             | p | r | t | - |
%                             -----------------
%                        -----------------        /
%                        | i | m | - | - |       /
%                        |---------------|      /
%                        | l | n | - | - |     / 
%                        -----------------    /
%              |    -----------------        / S=3
%              |    | a | c | e | g |       /
%          L=2 |    |---------------|      /
%              |    | b | d | f | h |     /
%              |    -----------------    /
%      
%                   -----------------
%                          T=4
%
%   in this case L=2, T=4 and S=3. Four trials are availabe for the first
%   stimulus, two for the second and three for the third stimulus. We have
%   Nt = [4; 3; 2] and maxNt = 4 and thus
%
%         ind = tridxs([4; 3; 2], 4, 3)
% 
%         ind =
% 
%              1
%              2
%              3
%              4
%              5
%              6
%              7
%              9
%             10
%
%   The indexes computed this way can be used to extract all the responses
%   corresponding to trials by means of the command
%         
%          R(:,ind) = 
%
%              a      c      e      g      i      m      o      q      s
%              b      d      f      h      l      n      p      r      t
%
% REMARKS
%   The inputs are obviously redundant since all the information required for
%   the computation is contained in NT. However, by passing the additional
%   arguments we can speed up things a bit.

% Copyright (C) 2010 Cesare Magri
% Version: 1.0.0

% -------
% LICENSE
% -------
% This software is distributed free under the condition that:
%
% 1. it shall not be incorporated in software that is subsequently sold;
%
% 2. the authorship of the software shall be acknowledged and the following
%    article shall be properly cited in any publication that uses results
%    generated by the software:
%
%      Magri C, Whittingstall K, Singh V, Logothetis NK, Panzeri S: A
%      toolbox for the fast information analysis of multiple-site LFP, EEG
%      and spike train recordings. BMC Neuroscience 2009 10(1):81;
%
% 3.  this notice shall remain in place in each source file.

% ----------
% DISCLAIMER
% ----------
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
% IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
% THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
% PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
% CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
% EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
% PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
% PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
% LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
% NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
% SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

one2maxNt = (1:maxNt);
ind = find((one2maxNt(ones(Ns,1),:)<=Nt(:, ones(maxNt,1))).');